
SynthVoicerNode {
	// one voice of a Voicer
	// this one handles standard synthdefs
	// for other types, subclass SynthVoicerNode and override methods
	// H. James Harkins, jamshark70@dewdrop-world.net

	var	isPlaying = false,
	<>isReleasing = false,
	<reserved = false,
	<>frequency,				// so voicer can identify notes to release
	<>lastTrigger = 0,			// ditto -- time of last trigger
	<releaseTime,				// expected release time, if known
	<>target, <>addAction,		// for allocating nodes
	<>bus,				// output bus to pass to things
	<synth, 		// the node
	<initArgs,	// send to each node on initiation
	<initArgDict,	// quicker access to initial arg values
	<defname,
	voicer,		// to help with globally mapped controls; what if voicer is nil?
	<myLastLatency,	// because latency is now variable at the voicer level
	// important because you may have 2 processes with different latencies
	// using the same Voicer
	<>steal = true,		// by default, if another note needs this object, its synth node can be killed
	// false means let the node die on its own (you lose synth activity control)
	<>id = 0;  // a layer ID for articulation

	*new { arg thing, args, bus, target, addAction = \addToTail, voicer, defname;
		target = target.asTarget;		// if nil, gives default server
		^super.new.init(thing, args, bus, target, addAction, voicer, defname)
	}

	init { arg th, ar, b, targ, addAct, par;
		synth.notNil.if({ this.free });		// if re-initing, drop synth node
		target = targ;		// save arguments
		addAction = addAct;

		// remove arg pairs from ar whose key is \freq, \gate or \outbus
		initArgs = this.makeInitArgs(ar);
		initArgDict = this.makeInitArgDict(initArgs);

		voicer = par;
		defname = th;

		// use given bus or hardware output
		bus = b ? Bus.new(\audio, 0, 1, target.server);
		^this
	}

	dtor {
		this.free
	}

	makeInitArgs { arg ar;
		var out;
		ar.notNil.if({
			out = Array.new;
			ar.pairsDo({ |name, value|
				if(#[\freq, \freqlag, \gate, \t_gate, \out, \outbus].includes(name.asSymbol).not
					// Buffers are 'noncontrol', per crucial-library
					and: { #[noncontrol, scalar].includes(value.rate) }
				) {
					out = out ++ [name.asSymbol, value];
				};
			});
		}, {
			out = Array(initArgDict.size * 2);
			initArgDict.keysValuesDo({ |name, value| out.add(name).add(value) });
		});
		^out
	}

	makeInitArgDict { |initArgs|
		var	out = IdentityDictionary.new;
		initArgs.pairsDo({ |name, value| out.put(name, value) });
		^out
	}

	initArgAt { |name| ^initArgDict[name.asSymbol] }

	// this test is split out of the above in case future subclasses need a different test
	// testArgClass { |argValue| ^argValue.asTestUGenInput.isValidSynthArg }

	triggerMsg { arg freq, gate = 1, args;
		var bundle, args2;
		// create osc message
		bundle = List.new;
		// assemble arguments
		args2 = initArgs ++ [\gate, gate, \t_gate, gate];
		// an arg could be a one-dimensional array
		// but it shouldn't have more dimensions than that
		args = (args ? []);
		(1, 3 .. args.size-1).do { |i|
			if(args[i].respondsTo(\flat)) { args[i] = args[i].flat };
		};
		(args.at(0).notNil).if({ args2 = args2 ++ args });
		freq.notNil.if({ args2 = args2 ++ [\freq, freq] });
		// make synth object
		synth = Synth.basicNew(this.asDefName, target.server);
		bundle.add(synth.newMsg(target, args2.asOSCArgArray ++ this.mapArgs
			++ [\out, bus.index, \outbus, bus.index], addAction));
		^bundle
	}

	triggerCallBack { ^nil }	// this is what OSCSchedule uses for its clientsidefunc
	// InstrVoicerNode uses this

	trigger { arg freq, gate = 1, args, latency;
		var bundle, watcher;
		if(freq.isValidVoicerArg) {
			this.shouldSteal.if({
				this.stealNode(synth, latency);
			});
			bundle = this.triggerMsg(freq, gate, args);
			target.server.listSendBundle(myLastLatency = latency, bundle);
			// 'this' would exist in susPedalNodes if it was released while susPedal = on
			// if we re-trigger it during that time, it's no longer 'released'
			// so we must remove it from the susPedalNodes collection
			voicer.susPedalNodes.remove(this);
			NodeWatcher.register(synth);
			// when the synth node dies, I need to set my flags
			watcher = SimpleController(synth)
			.put(\n_end, { |syn|
				// synth may have changed
				if(syn == synth) {
					reserved = isPlaying = isReleasing = false;
					synth = nil;
					this.releaseTime = nil;
				};
				watcher.remove;
			});
			frequency = freq;	// save frequency for Voicer.release
			lastTrigger = SystemClock.seconds;	// save time
			this.isPlaying = true;
			isReleasing = false;
		} {
			reserved = false;
		}
	}

	// triggerByEvent { |freq, gate(1), args, latency|
	// 	^this.trigger(freq, gate, args, latency);
	// }

	shouldSteal {
		^steal and: {
			isPlaying or: {
				synth.notNil and: {
					synth.isPlaying or: {
						// you might trigger a node, then very quickly trigger it again,
						// within the latency window. In that case, the 'synth' might not
						// *currently* be playing, but it will be playing by the time
						// the second .trigger should happen in the server.
						// but 'synth' should be non-nil in that case
						SystemClock.seconds - lastTrigger < (myLastLatency ? 0)
					}
				}
			}
		}
	}

	// must pass in (synth) node because, when a node is stolen, my synth variable has changed
	// to the new node, not the old one that should go away
	stealNode { |node, latency|
		// [synth, node, node.isPlaying, voicer.nodes.count(_.isPlaying)].debug("stealing: new synth, old synth, old is playing, number of playing voicer nodes");
		(synth.notNil/* and: { synth.isPlaying }*/).if({
			node.server.sendBundle(latency, #[error, -1], node.setMsg(\gate, -1.025), #[error, -2]);
		});
	}

	releaseTime_ { |seconds|
		if(seconds.isNil or: { (seconds < inf).not }) {
			releaseTime = nil  // inf releaseTime --> nil
		} {
			releaseTime = seconds
		}
	}

	releaseMsg { arg gate = 0;
		^[#[error, -1], [15, synth.nodeID, \gate, gate], #[error, -2]]
	}

	releaseCallBack {
		^nil
	}

	// release using Env's releaseNode
	// freq argument is because scheduled releases may be talking to a node that's been stolen.
	// In that case, the frequency will be different and the release should not happen.
	// if left nil, the release will go ahead.
	release { arg gate = 0, latency, freq;
		this.shouldRelease(freq).if({
			synth.server.listSendBundle(latency, this.releaseMsg(gate));
			this.isPlaying = false;
			isReleasing = true;
			this.releaseTime = nil;
			id = 0;
		});
	}

	// releaseByEvent { |gate(0), latency, freq|
	// 	^this.release(gate, latency, freq);
	// }

	isPlaying { ^isPlaying or: { (synth.notNil and: { synth.isPlaying }) } }
	isPlaying_ { |bool = false|
		isPlaying = reserved = bool;
	}
	reserved_ { |bool = false|
		reserved = bool;
		if(bool) { lastTrigger = SystemClock.seconds };
	}

	shouldRelease { arg freq;
		^(this.isPlaying and: { freq.isNil or: { freq == frequency } })
	}

	releaseNow { arg sec = 0.008;	// release immediately using linear decay
		this.release(sec.abs.neg - 1);	// -1 = instant decay, -0.5-1 = -1.5 = .5 sec decay
	}

	freeMsg {
		^[[11, synth.nodeID]]
	}

	freeCallBack { ^nil }

	free {	// remove from server; assumes envelope is already released
		(this.isPlaying).if({ synth.free; });
		this.isPlaying = false;
		id = 0;
	}

	setMsg { arg args;
		var ar, bundle;
		this.isPlaying.if({
			// ignore global controls (handled by Voicer.set)
			args = (args ? []).clump(2)
			.select({ arg a; voicer.globalControls.at(a.at(0).asSymbol).isNil })
			.flatten(1);
			^[[15, synth.nodeID] ++ args.asOSCArgArray]
		}, {
			^nil
		});
	}

	setCallBack { ^nil }

	set { arg args, latency;
		(this.isPlaying).if({
			target.server.listSendBundle(latency, this.setMsg(args));
		});
	}

	setArgDefaults { |args|
		args.pairsDo({ |key, value| initArgDict.put(key, value) });
		initArgs = this.makeInitArgs;
	}

	// nil if SynthDesc not found
	getSynthDesc { |synthLib|
		^(synthLib ?? { SynthDescLib.global }).tryPerform(\at, this.asDefName.asSymbol)
	}
	// 24-0831 new feature: allow an event to override the SynthVoicerNode's defname
	asDefName {
		var instr = \instrument.envirGet;
		^if(instr.notNil and: { instr != \default }) { instr } { defname }
	}

	// GENERAL SUPPORT METHODS
	server { ^target.server }	// tell the outside world where I live

	trace { this.isPlaying.if({ synth.trace }) }

	map { arg name, bus;	// do mapping for this node
		synth.notNil.if({
			synth.map(name, bus);
		});
	}

	mapArgsMsg { // assumes synth is loaded
		var mapMsg;
		// if nothing's in the globalControls dictionary, no need to do anything
		(voicer.globalControls.size > 0).if({
			mapMsg = [14, synth.nodeID];	// message header
			voicer.globalControls.keysValuesDo({ arg name, gc;
				mapMsg = mapMsg ++ [name, gc.bus.index];  // yep, add it to msg
			});
			^[mapMsg]
		});
		^nil		// if nothing to map
	}

	mapArgs { // assumes synth is loaded
		var out;
		// if nothing's in the globalControls dictionary, no need to do anything
		(voicer.globalControls.size > 0).if({
			out = Array.new(voicer.globalControls.size * 2);
			voicer.globalControls.keysValuesDo({ arg name, gc;
				out.add(name);
				out.add(("c" ++ gc.bus.index).asSymbol);
			});
			^out
		});
		^nil		// if nothing to map
	}

	displayName { ^defname }
}

InstrVoicerNode : SynthVoicerNode {
	// children are InstrVoicerNodes for any sub-patches
	var	<patch, <instr, <children;

	init { arg th, ar, b, targ, addAct, par, olddefname;
		var	def;
		patch.notNil.if({ this.free });		// if re-initing, drop synth node
		target = targ;		// save arguments
		instr = th;
		voicer = par;
		addAction = \addToTail;		// must always be so for compound patches
		// use given bus or hardware output
		bus = b ? Bus.new(\audio, 0, 1, target.server);
		olddefname.isNil.if({
			patch = this.makePatch(instr, ar);
			// cracky workaround for cxx's synthdef naming problem
			// this breaks nested patches but I never use that anyway
			def = patch.asSynthDef;
			def.name = (defname = def.name ++ UniqueID.next);
			// this might not work with wrapped Instr's
			try {
				def.perform(if(SynthDef.findRespondingMethodFor(\add).notNil)
					{ \add } { \memStore })
			} { |error|
				error.notNil.if({
					"Error occurred during InstrVoicerNode initialization: memStore.\nSending synthdef normally.".warn;
					error.reportError;
					"\nContinuing. Voicer will be usable. Pattern arguments will not be detected automatically.".postln;
					def.send(target.server);
				});
			};
		}, {
			// olddefname was not nil: the patch was made earlier by another node
			// this one will use the same synthdef
			defname = olddefname;
			initArgs = this.makeInitArgs(ar);
		});
		initArgDict = this.makeInitArgDict(initArgs);
	}

	dtor {
		super.dtor;	// release synth nodes
		if(patch.notNil) {
			// if so, the I made the synthdef and I should discard it
			target.server.sendMsg(\d_free, defname);
			SynthDescLib.global.removeAt(defname.asSymbol);
			// do I need to remove from the abstractplayer cache?
		};
		patch.free;	// garbage collect patch
		patch = nil;
	}

	makePatch { |instr, args|
		var class = instr.tryPerform(\patchClass) ?? { Patch },
		patchArgs = this.makePatchArgs(instr, args);
		^class.new(instr, patchArgs)
	}

	// does trigger et al. need to hit the children?
	trigger { arg freq, gate = 1, args, latency;
		var bundle;
		if(freq.isValidVoicerArg) {
			this.shouldSteal.if({
				this.stealNode(synth, latency);
			});
			bundle = bundle ++ this.triggerMsg(freq, gate, args);
			target.server.listSendBundle(myLastLatency = latency, bundle);
			voicer.susPedalNodes.remove(this);

			frequency = freq;
			lastTrigger = SystemClock.seconds;
		} {
			reserved = false;
		}
	}

	triggerMsg { arg freq, gate = 1, args;
		var bundle, watcher;
		bundle = Array.new;
		// make messages for children
		children.do({ arg child; bundle = bundle ++ child.triggerMsg(freq, gate, args); });

		bundle = bundle ++ super.triggerMsg(freq, gate, args);
		// super.triggerMsg also handles global mapping
		NodeWatcher.register(synth);  // we now have a synth object too
		// when the synth node dies, I need to set my flags
		watcher = SimpleController(synth)
		.put(\n_end, { |syn|
			// synth may have changed
			if(syn == synth) {
				reserved = isPlaying = isReleasing = false;
				synth = nil;
				this.releaseTime = nil;
			};
			watcher.remove;
		});
		this.isPlaying = true;
		isReleasing = false;
		^bundle
	}

	triggerCallBack { ^nil }	// patch updating can be ignored

	release { arg gate = 0, latency, freq;
		this.shouldRelease(freq).if({
			this.target.server.listSendBundle(latency, this.releaseMsg(gate));
			this.isPlaying = false;
			isReleasing = true;
			releaseTime = nil;
			id = 0;
		});
	}

	// releaseMsg assumes that the caller has done all the safety checks
	releaseMsg { arg gate = 0, wrap = true;	// wrap with \error msg?
		var bundle;
		(synth.notNil).if({
			wrap.if({
				bundle = [#[\error, -1]];
			}, {
				bundle = Array.new;
			});
			children.do({ |child| bundle = bundle ++ child.releaseMsg(gate, false); });
			bundle = bundle ++ [[15, synth.nodeID, \gate, gate]];
			wrap.if({ bundle = bundle ++ [#[error, -2]]; });
		})
		^bundle
	}

	releaseCallBack { arg gate;
		^nil
	}

	freeMsg {
		var bundle;
		(synth.notNil and: { synth.isPlaying }).if({
			this.isPlaying = false;
			bundle = List.new;
			// collect free messages for children
			children.do({ arg child; bundle = bundle ++ child.freeMsg; });
			^bundle ++ [[11, synth.nodeID]];
		}, {
			^[]	// if synth isn't playing, freeMsg is meaningless
		});
	}

	freeCallBack {
		^nil
	}

	set { arg args, latency;
		synth.notNil.if({
			target.server.listSendBundle(latency, this.setMsg(args));
		});
		this.setCallBack.value(args);
	}

	setMsg { arg args;
		var bundle, ar, argColl;
		synth.notNil.if({
			bundle = Array.new;
			// collect set messages for children
			children.do({ arg child;
				bundle = bundle ++ child.setMsg(args)
			});

			^bundle ++ super.setMsg(args)
		}, {
			^[]	// if synth isn't playing, setMsg is meaningless
		});
	}

	setCallBack {
		^nil
	}

	free {
		var bundle;
		target.server.listSendBundle(nil, this.freeMsg);
		this.isPlaying = false;
		id = 0;
	}

	displayName { ^instr.name.asString }

	// PRIVATE

	mapArgsMsg { 		// collects mapArgsMsgs for this and children
		var bundle;
		bundle = Array.new;
		children.do({ arg child; bundle = bundle ++ child.mapArgsMsg });
		bundle = bundle ++ super.mapArgsMsg;	// use SynthVoicerNode.mapArgsMsg for the meat
		^bundle
	}

	map { arg name, bus;
		children.do({ arg child; child.map(name, bus) });
		synth.notNil.if({
			synth.map(name, bus);
		});
	}

	makePatchArgs { arg instr, ar;
		var	argNames, argSpecs, argArray, proto,
		argIndex, gateIndex, thisArg, basePatch, temp;

		// to support instr wrapping, I need to know what args are created during def building
		try {
			// can throw this one away
			(basePatch = (instr.tryPerform(\patchClass) ?? { Patch }).new(instr)).asSynthDef;
			argNames = basePatch.argNames;
			argSpecs = basePatch.argSpecs;
		} {
			argNames = instr.func.def.argNames;
			argSpecs = instr.specs;
		};
		basePatch.free;	// remove dependents
		initArgs = Array.new;

		// if no args, make empty array
		ar = ar ? [];
		argArray = argNames.collect({ arg name, i;
			argIndex = ar.indexOf(name);	// find specified value, if any
			thisArg = argIndex.isNil.if({ nil }, { ar.at(argIndex+1) });

			switch(name)
			{ \gate } { KrNumberEditor(thisArg ? 0, argSpecs[i]).lag_(nil) }
			{ \t_gate } { SimpleTrigger(argSpecs[i]) }

			{		// if you're nesting a patch, and the inner patch has a gate arg,
				// and the outer one does not, use inner patch as triggerable
				// I may remove this support because it never worked well
				(thisArg.isKindOf(Instr)).if({
					// make inner patch
					proto = InstrVoicerNode(thisArg, ar.at(argIndex+2),
						bus, target, addAction, voicer);
					children = children.add(proto);
					proto.patch	// output new Patch as arg
				}, {
					case
					// once upon a time, I didn't need this special case
					{ thisArg.isKindOf(NumberEditor) } { thisArg }
					{ thisArg.isNumber.not } {
						thisArg.dereference	// Refs of SimpleNumbers for fixed args
					}
					{		// otherwise make a default control (see Patch-createArgs)
						proto = argSpecs.at(i).defaultControl;
						proto.tryPerform('spec_',argSpecs.at(i)); // make sure it does the spec
						argIndex.notNil.if({
							proto.tryPerform('value_', thisArg);// set its value
						});
						// so all nodes are properly initialized at play time
						// only SimpleNumber args need to be added here; others
						// will be fixed args that we can't talk to
						(argIndex.notNil
							and: { #[\freq, \freqlag, \gate, \t_gate, \out]
								.includes(name).not })
						.if({
							initArgs = initArgs ++ [name, thisArg];
						});
						proto
					};
				});
			};
		});

		^argArray
	}

}

MIDIVoicerNode : SynthVoicerNode {
	classvar fakeSynthDesc;
	var midichannel = 0, lastVelocity, noteOffMsg;
	var noteFunc;

	*initClass {
		var cn = [
			ControlName(\freq, 0, \control, 440),
			ControlName(\amp, 1, \control, 0.5),
			ControlName(\acc, 2, \control, 0),
			ControlName(\accAmt, 3, \control, 0.3)
		];
		var cd = IdentityDictionary.new;
		cn.do { |cn| cd.put(cn.name, cn) };
		fakeSynthDesc = SynthDesc()
		.controls_(cn)
		.controlNames_(cn.collect(_.name))
		.controlDict_(cd);
	}

	*new { arg thing, args, voicer;
		// note: voicer arg is called 'voicer' in the superclass
		// do not try to match the keyword to the init method
		^super.new(thing, args, voicer: voicer)  // super calls my own init
	}

	// most are ignored
	init { |th, ar, b, targ, addAct, par|
		var chanIndex;
		voicer = par;
		noteFunc = { |note| note };
		initArgDict = IdentityDictionary[\accAmt -> 0.2];
		ar.tryPerform(\pairsDo) { |key, value|
			switch(key)
			{ \chan } {
				initArgDict.put(key, value);
				midichannel = value;
			}
			{ \int } {
				initArgDict.put(key, value);
				if(value.asBoolean) {
					noteFunc = { |note| note.round.asInteger }
				}
			}
			{ \accAmt } {
				initArgDict.put(key, value);
			}
		};
		// this object handles note messages only
		defname = MIDINoteMessage(
			channel: midichannel, device: th,
			latency: Server.default.latency
		);
		noteOffMsg = MIDINoteMessage(
			velocity: 0,
			channel: midichannel, device: th,
			latency: Server.default.latency
		);
		lastVelocity = 64;
	}

	trigger { arg freq, gate = 1, args, latency;
		var bundle;
		var acc = 0, accAmt = 1;
		if(freq.isValidVoicerArg) {
			if(this.shouldSteal) {
				this.stealNode(frequency, latency);
			};
			if(args.size > 1) {
				#acc, accAmt = args.findPairKeys(#[acc, accAmt], #[0, 1]);
			};
			if(acc > 0) { gate = (gate * (accAmt + 1)).clip(0, 1) };
			defname.play(noteFunc.(freq.cpsmidi), (gate * 127).asInteger);
			// 'this' would exist in susPedalNodes if it was released while susPedal = on
			// if we re-trigger it during that time, it's no longer 'released'
			// so we must remove it from the susPedalNodes collection
			voicer.susPedalNodes.remove(this);
			frequency = freq;	// save frequency for Voicer.release
			lastTrigger = SystemClock.seconds;	// save time
			this.isPlaying = true;
			isReleasing = false;
		} {
			reserved = false;
		}
	}

	shouldSteal {
		^steal and: {
			isPlaying or: {
				// not sure
				SystemClock.seconds - lastTrigger < (myLastLatency ? 0)
			}
		}
	}

	// must pass in (synth) node because, when a node is stolen, my synth variable has changed
	// to the new node, not the old one that should go away
	stealNode { |freq, latency|
		if(freq.notNil) {
			noteOffMsg.play(noteFunc.(freq.cpsmidi));
		}
	}

	// releaseTime_ {}

	releaseCheckNote { |oldNote|
		^voicer.nodes.every { |node|
			node === this or: {
				node.isPlaying.not or: { (noteFunc.(node.frequency.cpsmidi) != oldNote) }
			}
		}
	}

	releaseCallBack {
		^nil
	}

	release { arg gate = 0, latency, freq;
		var num;
		if(this.shouldRelease(freq)) {
			freq = freq ?? { frequency };
			num = noteFunc.(freq.cpsmidi);
			if(this.releaseCheckNote(num)) {
				noteOffMsg.play(num);
			};
			this.isPlaying = false;
			isReleasing = true;
			this.releaseTime = nil;
			id = 0;
		};
	}

	isPlaying { ^isPlaying }

	reserved_ { |bool = false|
		reserved = bool;
		// for some reason that I forget, a MIDIVoicerNode needs to have
		// a synth to be fully reserved -- but there's no server node for it
		// so 0xFFFFFFFF prevents the server's node allocator from wasting a node ID for it
		if(bool) { synth = Synth.basicNew(\dummy, Server.default, 0xFFFFFFFF) };
	}

	freeMsg {
		^this.releaseMsg(0)
	}

	free {	// stop node; in Rack, can do this only by closing the gate
		if(this.isPlaying) { this.release(0, freq: frequency) };
		this.isPlaying = false;
		id = 0;
	}

	setMsg {}
	setCallBack { ^nil }
	// by MIDI, can only set midinote
	// maybe later do mapped args to CCs but not today
	set { |args, latency|
		var i, j, note, vel, oldNote;
		if(this.isPlaying) {
			i = args.detectIndex(_ == \freq);
			if(i.notNil) {
				oldNote = noteFunc.(frequency.cpsmidi);
				note = noteFunc.(args[i+1].cpsmidi);
				if(note != oldNote) {
					frequency = args[i+1];
					j = args.detectIndex(_ == \gate);
					if(j.notNil) {
						vel = (args[j+1] * 127).round.asInteger;
						lastVelocity = vel;
					} {
						vel = lastVelocity;
					};
					defname.play(note, vel);
					SystemClock.sched(0.01, {
						if(this.releaseCheckNote(oldNote)) {
							noteOffMsg.play(oldNote);
						};
					});
				};
			};
		};
	}
	setArgDefaults {}
	getSynthDesc { ^fakeSynthDesc }

	// not really applicable but something upstream will complain if I don't...
	server { ^Server.default }

	trace {}
	map {}
	mapArgsMsg {}
	mapArgs {}
	displayName { ^defname.asString }
}

SynVoicerNode : SynthVoicerNode {
	var newMethod = \basicNew;

	usePaths { ^newMethod == \basicNewByArgPaths }
	usePaths_ { |bool|
		newMethod = #[basicNew, basicNewByArgPaths][bool.binaryValue]
	}

	triggerMsg { arg freq, gate = 1, args;
		var args2, gcs;
		var plugKey, plug;
		var fixPlug = { |args, key, value, i|
			// in case we're in an event
			plugKey = (key.asString ++ "Plug").asSymbol;
			plug = plugKey.envirGet;
			if(plug.canMakePlug) {
				args[i+1] = plug.dereference.valueEnvir(value);
			};
		};
		// assemble arguments
		args2 = initArgs ++ [\gate, gate, \t_gate, gate];
		// an arg could be a one-dimensional array
		// but it shouldn't have more dimensions than that
		args = (args ? []);
		args.pairsDo { |key, value, i|
			if(value.respondsTo(\flat)) { args[i+1] = value.flat };
			// avoid duplicating Plugs
			if(initArgDict[key].notNil and: {
				value === initArgDict[key]
			}) {
				args[i] = nil;
				args[i+1] = nil;
			} {
				fixPlug.(args, key, value, i);
			};
		};

		(args.notEmpty).if({ args2 = args2 ++ args.select(_.notNil) });
		// this may need to change for freq plugs
		freq.notNil.if({ args2 = args2 ++ [\freq, freq] });
		fixPlug.(args2, \freq, freq, args2.size - 2);

		// experimental 24-0419: are there any plugs for GCs?
		gcs = this.mapArgs;
		gcs.pairsDo { |key, value, i|
			fixPlug.(gcs, key, value, i);
		};

		args2 = args2 ++ gcs ++ [\out, bus.index, \outbus, bus.index];
		// make synth object
		synth = Syn.perform(newMethod, this.asDefName, args2, target, addAction);
		// note, no multichannel expansion here
		// mc-expansion is handled in voicerNote events
		^synth.prepareToBundle;
	}

	triggerCallBack { ^nil }	// this is what OSCSchedule uses for its clientsidefunc
	// InstrVoicerNode uses this

	trigger { arg freq, gate = 1, args, latency;
		var bundle, watcher;
		if(freq.isValidVoicerArg) {
			this.shouldSteal.if({
				this.stealNode(synth, latency);
			});
			bundle = this.triggerMsg(freq, gate, args);
			synth.sendBundle(bundle, myLastLatency = latency);
			// 'this' would exist in susPedalNodes if it was released while susPedal = on
			// if we re-trigger it during that time, it's no longer 'released'
			// so we must remove it from the susPedalNodes collection
			voicer.susPedalNodes.remove(this);
			watcher = SimpleController(synth)
			.put(\didFree, { |syn|
				if(syn === synth) {
					reserved = isPlaying = isReleasing = false;
					synth = nil;
					this.releaseTime = nil;
				};
				watcher.remove;
			});
			frequency = freq;	// save frequency for Voicer.release
			lastTrigger = SystemClock.seconds;	// save time
			this.isPlaying = true;
			isReleasing = false;
		} {
			reserved = false;
		}
	}

	stealNode { |node, latency|
		synth.notNil.if({
			this.releaseMsg(-1.025).sendOnTime(node.server, latency)
		});
	}

	releaseMsg { arg gate = 0;
		var bundle = OSCBundle.new;
		bundle.add(#[error, -1]);
		synth.releaseToBundle(bundle, gate);
		bundle.add(#[error, -2])
		^bundle
	}

	// release using Env's releaseNode
	// freq argument is because scheduled releases may be talking to a node that's been stolen.
	// In that case, the frequency will be different and the release should not happen.
	// if left nil, the release will go ahead.
	release { arg gate = 0, latency, freq;
		this.shouldRelease(freq).if({
			this.releaseMsg(gate).sendOnTime(synth.server, latency);
			this.isPlaying = false;
			isReleasing = true;
			this.releaseTime = nil;
			id = 0;
		});
	}

	freeMsg {
		^synth.freeToBundle
	}

	free {	// remove from server; assumes envelope is already released
		(this.isPlaying).if({ synth.free });
		this.isPlaying = false;
		id = 0;
	}

	set { arg args, latency;
		(this.isPlaying).if({
			this.setMsg(args).sendOnTime(target.server, latency);
		});
	}
	setMsg { arg args;
		var ar;
		this.isPlaying.if({
			// ignore global controls (handled by Voicer.set)
			args = (args ? []).clump(2)
			.select({ arg a;
				// no GCs for set
				voicer.globalControls.at(a[0].asSymbol).isNil
				// and no Plugs (maybe fix later)
				and: { a[1].isKindOf(Plug).not }
			})
			.flatten(1);
			^synth.setToBundle(nil, *args)
		}, {
			^nil
		});
	}

	map { arg name, bus;	// do mapping for this node
		synth.notNil.if({
			// Syn doesn't (yet?) do .map directly
			// for some reason I can't recall, I'm passing in
			// only the bus number, not the Bus object... :|
			synth.set(name, "c" ++ bus);
		});
	}

	makeInitArgs { arg ar;
		var out;
		ar.notNil.if({
			out = Array.new;
			ar.pairsDo({ |name, value|
				if(#[\freq, \freqlag, \gate, \t_gate, \out, \outbus].includes(name.asSymbol).not
					// Buffers are 'noncontrol', per crucial-library
					and: {
						#[noncontrol, scalar].includes(value.rate)
						or: { value.isKindOf(Plug) }
					}
				) {
					out = out ++ [name.asSymbol, value];
				};
			});
		}, {
			out = Array(initArgDict.size * 2);
			initArgDict.keysValuesDo({ |name, value| out.add(name).add(value) });
		});
		^out
	}
}
